using System;
using System.Collections.Generic;
using System.ComponentModel.DataAnnotations;
using System.ComponentModel.DataAnnotations.Schema;
using Microsoft.EntityFrameworkCore;

/*
Ukoliko se podrazumevana pravila ne prate potrebno je dodatno opisati kako se domenske klase mapiraju
na tabele u bazi podataka. Drugi nacin da se to ostvari jeste koriscenjem Fluent API u metodi OnModelBuilding
klase izvedene iz DbContext klase.
*/
namespace UnconventionalModelWithFluentAPI {

    public class Korisnik {
        public int kljuc { get; set; }
        public string ime { get; set; }
        public string prezime { get; set; }
        public string email { get; set; }
        public string pol { get; set; }
        public string korisnickoIme { get; set; }
        public string lozinka { get; set; }

        /*
        One-to-many i Many-to-many veze i dalje zahtevaju kolekcije.
        */
        public ICollection<ImaUlogu> imaUloguList { get; set; }
        public ICollection<Akcija> akcijaList { get; set; }

        public override string ToString ( ) {
            return "User(" + this.kljuc + "): " + this.ime + ", " + this.prezime + ", " + this.email + ", " + this.pol + ", " + this.korisnickoIme + ", " + this.lozinka;
        }
    }

    public class Uloga {
        public int kljuc { get; set; }
        public string ime { get; set; }
        public ICollection<ImaUlogu> imaUloguList { get; set; }
        public override string ToString ( ) {
            return "Role(" + this.kljuc + "): " + this.ime;
        }
    }

    public class ImaUlogu {
        public int korisnikId { get; set; }
        public Korisnik korisnik { get; set; }
        public int ulogaId { get; set; }
        public Uloga uloga { get; set; }
        public override string ToString ( ) {
            return "HasRole(" + this.korisnikId + ", " + this.ulogaId + ")";
        }
    }

    public class Akcija {
        public int kljuc { get; set; }
        public int korisnikId  { get; set; }
        public Korisnik korisnik { get; set; }
        public string ime { get; set; }
        public DateTime vreme { get; set; }

        public override string ToString ( ) {
            return "Action(" + this.kljuc + "): " + this.korisnikId + ", " 
                                             + this.ime + ", " 
                                             + this.vreme;
        }
    }
    public class UnconventionalContextWithFluentAPI : DbContext {
        // Imena tabela u ovom slucaju ne moraju da odgovaraju imenima DbSet polja.
        public DbSet<Korisnik> korisnici { get; set; }
        public DbSet<Uloga> uloge { get; set; }
        public DbSet<Akcija> akcije { get; set; }
        public DbSet<ImaUlogu> imaUlogu { get; set; }

        protected override void OnConfiguring ( DbContextOptionsBuilder builder ) {
            builder.UseMySQL ( "Server=localhost;Database=user_database;Uid=root;Pwd=root;" );
        }

        protected override void OnModelCreating ( ModelBuilder builder ) {
            // Povezivanje klase Korisnik
            builder.Entity<Korisnik> ( ).ToTable ( "users" );             // Tabela na koju se klasa odnosi
            builder.Entity<Korisnik> ( ).HasKey ( k => new { k.kljuc } ); // Sta predstavlja primarni kljuc tabele
            builder.Entity<Korisnik> ( ).Property ( k => k.kljuc )        // Podesavanje svakog polja klase
                                        .HasColumnName ( "id" )
                                        .IsRequired ( );
            builder.Entity<Korisnik> ( ).Property ( k => k.ime )                                       
                                        .HasColumnName ( "firstName" )
                                        .HasColumnType ( "varchar(50)")
                                        .IsRequired ( );
            builder.Entity<Korisnik> ( ).Property ( k => k.prezime )                                       
                                        .HasColumnName ( "lastName" )
                                        .HasColumnType ( "varchar(50)")
                                        .IsRequired ( );
            builder.Entity<Korisnik> ( ).Property ( k => k.email )                                       
                                        .HasColumnName ( "email" )
                                        .HasColumnType ( "varchar(50)")
                                        .IsRequired ( );
            builder.Entity<Korisnik> ( ).Property ( k => k.pol )                                       
                                        .HasColumnName ( "gender" )
                                        .HasColumnType ( "varchar(50)")
                                        .IsRequired ( );
            builder.Entity<Korisnik> ( ).Property ( k => k.korisnickoIme )                                       
                                        .HasColumnName ( "username" )
                                        .HasColumnType ( "varchar(50)")
                                        .IsRequired ( );
            builder.Entity<Korisnik> ( ).Property ( k => k.lozinka )                                       
                                        .HasColumnName ( "password" )
                                        .HasColumnType ( "varchar(50)")
                                        .IsRequired ( );
            
            // Povezivanje klase Uloga 
            builder.Entity<Uloga> ( ).ToTable ( "roles" );
            builder.Entity<Uloga> ( ).HasKey ( u => u.kljuc );
            builder.Entity<Uloga> ( ).Property ( u => u.kljuc )                                       
                                        .HasColumnName ( "id" )
                                        .IsRequired ( );
            builder.Entity<Uloga> ( ).Property ( u => u.ime )                                       
                                        .HasColumnName ( "name" )
                                        .HasColumnType ( "varchar(50)")
                                        .IsRequired ( );

            // Povezivanje klase Akcija i podesavanje One-to-many veze
            builder.Entity<Akcija> ( ).ToTable ( "actions" );
            builder.Entity<Akcija> ( ).HasKey ( a => new { a.kljuc } );
            builder.Entity<Akcija> ( ).Property ( a => a.kljuc )
                                        .HasColumnName ( "id" )
                                        .IsRequired ( );
            builder.Entity<Akcija> ( ).Property ( a => a.korisnikId )
                                        .HasColumnName ( "userId" )
                                        .IsRequired ( );
            builder.Entity<Akcija> ( ).Property ( a => a.ime )
                                        .HasColumnName ( "name" )
                                        .HasColumnType ( "varchar(50)" )
                                        .IsRequired ( );
            builder.Entity<Akcija> ( ).Property ( a => a.vreme )
                                        .HasColumnName ( "timestamp" )
                                        .HasColumnType ( "datetime" )
                                        .IsRequired ( );            

            builder.Entity<Akcija> ( ).HasOne<Korisnik> ( a => a.korisnik )
                                        .WithMany ( k => k.akcijaList )
                                        .HasForeignKey ( a => a.korisnikId );

            // Povezivanje klase ImaUlogu i podesavanje Many-to-many veze
            builder.Entity<ImaUlogu> ( ).ToTable ( "hasRole" );
            builder.Entity<ImaUlogu> ( ).HasKey ( imaUlogu => new { imaUlogu.korisnikId, imaUlogu.ulogaId } );
            builder.Entity<ImaUlogu> ( ).Property ( i => i.korisnikId )
                                        .HasColumnName ( "userId" )
                                        .IsRequired ( );
            builder.Entity<ImaUlogu> ( ).Property ( i => i.ulogaId )
                                        .HasColumnName ( "roleId" )
                                        .IsRequired ( );

            builder.Entity<ImaUlogu> ( ).HasOne ( i => i.korisnik )
                                        .WithMany ( k => k.imaUloguList )
                                        .HasForeignKey ( i => i.korisnikId );

            builder.Entity<ImaUlogu> ( ).HasOne ( i => i.uloga )
                                        .WithMany ( u => u.imaUloguList )
                                        .HasForeignKey ( i => i.ulogaId );
        }
    }
}