
using System;
using System.Collections.Generic;
using Microsoft.EntityFrameworkCore;

/*
Postoje podrazumevana pravila na osnovu kojih EF Core zakljucuje kako se poveze tabele iz baze sa domenskim klasama.
*/
namespace ConventionalModel {
    /*
    Atributi entiteskih klasa moraju da odgovaraju kolonoma u bazi podatak i po imenu i po tipu.
    Preslikavanje SQL tipova u C# tipove dato je sledecom tabelom:
    C# tip      SQL tip 
    ====================
    int	        int
    string	    nvarchar(Max)
    decimal	    decimal(18,2)
    float	    real
    byte[]	    varbinary(Max)
    DateTime	datetime
    bool	    bit
    byte	    tinyint
    short	    smallint
    long	    bigint
    double	    float
    char	    No mapping
    sbyte	    No mapping (throws exception)
    object	    No mapping
    */
    public class User {
        public int id { get; set; }
        public string firstName { get; set; }
        public string lastName { get; set; }
        public string email { get; set; }
        public string gender { get; set; }
        public string username { get; set; }
        public string password { get; set; }
        /*
        One-to-many veze se realizuju tako sto se sa One strane doda polej klase koje predstavlja
        kolekciju objekata tabele koja predstavlja Many stranu, u ovom slucaju tabela "actions".
        */
        public ICollection<Action> actionList { get; set; }
        /*
        Many-to-many veze podrazumevaju postojanje tabele koja predstavlja vezu izmedju dve Many strane. Stoga,
        one se realizuju kao dve One-to-Many veze stime sto je neophodno dodatno navesti sta predstavlja kljuc
        vezne tebale u metodi OnModelConfiguring klase izvedene iz DbContext klase.
        */
        public ICollection<HasRole> hasRoleList { get; set; }

        public override string ToString ( ) {
            return "User(" + this.id + "): " + this.firstName + ", " + this.lastName + ", " + this.email + ", " + this.gender + ", " + this.username + ", " + this.password; 
        } 
    }

    public class Role {
        public int id { get; set; }
        public string name { get; set; }
        public ICollection<HasRole> hasRoleList { get; set; }

        public override string ToString ( ) {
            return "Role(" + this.id + "): " + this.name;
        }
    }

    public class HasRole {
        /*
        Vezne tabele u Many-to-many vezama predstavljaju Many stranu dve One-to-Many veze. Stoga, neophodno
        je dodati dve reference na Many stranu ove veze. Pored toga, neophodno je dodati dva celobrojna atributa
        koji predstavljaju migrirane primarne kljuceve dve tabele koje predstavljaju Many stranu. 
        */
        public int userId { get; set; }
        public User user { get; set; }
        public string roleId { get; set; }
        public Role role { get; set; }

        public override string ToString ( ) {
            return "HasRole(" + this.userId + ", " + this.roleId + ")";
        }
    }

    public class Action {
        /*
        Prilikom konfiguracije One-to-many na Many strani neophodno je dodati referencu na One stranu.
        Opciono, moze se dodati celobrojni atribut koji predstavlja migrirani primarni kljuc iz tabele 
        koja predstavlja One stranu.
        */
        public int id { get; set; }
        public int userId  { get; set; }
        public User user { get; set; }
        public string name { get; set; }
        public DateTime timestamp { get; set; }

        public override string ToString ( ) {
            return "Action(" + this.id + "): " + this.userId + ", " 
                                             + this.name + ", " 
                                             + this.timestamp;
        }
    }


    public class ConventionalContext : DbContext {
        // Imena tabela moraju da odgovaraju imenima DbSet polja.
        public DbSet<User> users { get; set; }
        public DbSet<Role> roles { get; set; }
        public DbSet<HasRole> hasRole { get; set; }
        public DbSet<Action> actions { get; set; }

        protected override void OnConfiguring ( DbContextOptionsBuilder builder ) {
            /* 
            Prilikom konfiguracije potrebno je navesti adresu servera baze podataka, ime baze i kredencijale
            sa kojima se pristupa bazi. Format string za povezivanje moze se naci na adresi https://www.connectionstrings.com/.
            */
            builder.UseMySQL ( "Server=localhost;Database=user_database;Uid=root;Pwd=root;" );
        }

        protected override void OnModelCreating ( ModelBuilder builder ) {
            /*
            Tabela HasRole ima kompozitni primarni kljuc. To je potrebno navesti u ovoj metodi.
            EF Core za razliku od EF ne podrazava pravljenje kompozitnog primarnog kljuca pomocu Key
            atributa.
            */
            builder.Entity<HasRole> ( ).HasKey ( hasRole => new { hasRole.userId, hasRole.roleId } );
        }
    }
}