pragma solidity ^0.8.18;

contract MultiSignatureWallet {
    event Deposit ( address indexed sender, uint256 amount, uint256 balance );
    event SubmitTransaction ( address indexed owner, uint256 indexed transaction_index, address indexed to, uint256 value );
    event ConfirmTransaction ( address indexed owner, uint256 indexed transaction_index );
    event RevokeTransaction ( address indexed owner, uint256 indexed transaction_index );
    event ExecuteTransaction ( address indexed owner, uint256 indexed transaction_index );

    address[] public owners;
    mapping ( address => bool ) public isOwner;
    uint256 public number_of_confirmations_required;
    uint256 available_amount;

    enum State {
        PENDING,
        EXECUTED,
        REVOKED
    }

    struct Transaction {
        address to;
        uint256 value;
        State state;
        uint256 number_of_confirmations;
    }

    // mapping from tx index => owner => bool
    mapping ( uint256 => mapping ( address => bool ) ) public is_confirmed;

    Transaction[] public transactions;

    modifier only_owner ( ) {
        require ( isOwner[msg.sender], "Not owner!" );
        _;
    }

    modifier transaction_exists ( uint256 _transaction_index ) {
        require ( _transaction_index < transactions.length, "Transaction does not exist!" );
        _;
    }

    modifier transaction_in_state ( uint256 _transaction_index, State state ) {
        require ( transactions[_transaction_index].state == state, "Transaction not in required state!" );
        _;
    }

    modifier not_confirmed ( uint256 _transaction_index ) {
        require ( !is_confirmed[_transaction_index][msg.sender], "Transaction already confirmed!" );
        _;
    }

    constructor ( address[] memory _owners, uint256 _number_of_confirmations_required ) {
        require ( _owners.length > 0, "owners required" );
        require (
            _number_of_confirmations_required > 0 && _number_of_confirmations_required <= _owners.length,
            "Invalid number of required confirmations!"
        );

        for ( uint256 i = 0; i < _owners.length; i++ ) {
            address owner = _owners[i];

            require ( owner != address(0), "Invalid owner!" );
            require ( !isOwner[owner], "Owner not unique!" );

            isOwner[owner] = true;
            owners.push ( owner );
        }

        number_of_confirmations_required = _number_of_confirmations_required;
    }

    function deposit ( ) external payable {
        available_amount += msg.value;

        emit Deposit ( msg.sender, msg.value, address ( this ).balance );
    }

    function submit_transaction ( address _to, uint256 _value ) 
        public 
        only_owner 
    {
        require ( _value < available_amount, "Insufficient funds!" );

        uint256 transaction_index = transactions.length;

        transactions.push (
            Transaction ({
                to: _to,
                value: _value,
                state: State.PENDING,
                number_of_confirmations: 0
            })
        );

        available_amount -= _value;

        emit SubmitTransaction ( msg.sender, transaction_index, _to, _value );
    }

    function confirm_transaction ( uint256 _transaction_index )
        public
        only_owner
        transaction_exists ( _transaction_index )
        transaction_in_state ( _transaction_index, State.PENDING )
        not_confirmed ( _transaction_index )
    {
        Transaction storage transaction = transactions[_transaction_index];
        transaction.number_of_confirmations += 1;
        is_confirmed[_transaction_index][msg.sender] = true;

        emit ConfirmTransaction ( msg.sender, _transaction_index );
    }

    function execute_transaction ( uint256 _transaction_index )
        public
        only_owner
        transaction_exists ( _transaction_index )
        transaction_in_state ( _transaction_index, State.PENDING )
    {
        Transaction storage transaction = transactions[_transaction_index];

        require (
            transaction.number_of_confirmations >= number_of_confirmations_required,
            "Cannot execute transaction!"
        );

        transaction.state = State.EXECUTED;

        payable ( transaction.to ).transfer ( transaction.value );

        emit ExecuteTransaction ( msg.sender, _transaction_index );
    }

    function revoke_transaction ( uint256 _transaction_index )
        public
        only_owner
        transaction_exists ( _transaction_index )
        transaction_in_state ( _transaction_index, State.PENDING )
    {
        Transaction storage transaction = transactions[_transaction_index];

        transaction.state = State.REVOKED;

        available_amount += transaction.value;

        emit RevokeTransaction ( msg.sender, _transaction_index );
    }

    function get_owners ( ) public view returns ( address[] memory ) {
        return owners;
    }

    function get_transaction_count ( ) public view returns ( uint256 ) {
        return transactions.length;
    }

    function get_transaction ( uint256 _transaction_index )
        public
        view
        returns (
            address to,
            uint256 value,
            State status,
            uint256 number_of_confirmations
        )
    {
        Transaction memory transaction = transactions[_transaction_index];

        return (
            transaction.to,
            transaction.value,
            transaction.state,
            transaction.number_of_confirmations
        );
    }
}
