// Prevođenje:
//    gcc kvm_zadatak1.c -o kvm_zadatak1
// Pokretanje:
//    ./kvm_zadatak1 guest
//
// Koristan link: https://www.kernel.org/doc/html/latest/virt/kvm/api.html
//
// Zadatak: Omogućiti ispravno izvršavanje gost asemblerskog programa. Gost program pristupa serijsom portu 0x3f8 (out instrukcija), pa
//          je potrebno emulirati serijski port 0x3f8 i parametar out instrukcije ispisati na standardnom izlazu. 
//          Pristup I/O portovima preko IN/OUT instrukcija izaziva VM izlazak.
// 

#include <fcntl.h>
#include <linux/kvm.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <unistd.h>

int main(int argc, char *argv[]) {
  uint8_t *mem = MAP_FAILED;
  int kvm_fd = -1;
  int vm_fd = -1;
  int vcpu_fd = -1;
  int ret;
  int stop = 0;
  FILE* img = NULL;
  size_t mem_size = 1 << 20; // 1 MiB
  size_t pc_val = 0x1000;
  struct kvm_run *run = MAP_FAILED;

  if (argc != 2) {
    printf("The program requests an image to run: %s <guest-image>\n", argv[0]);
    return 1;
  }

  // Sistemski poziv za otvaranje /dev/kvm
  // Povratna vrednost je deskriptor fajla
  kvm_fd = open("/dev/kvm", O_RDWR);
  if (kvm_fd == -1) {
    perror("Failed to open /dev/kvm\n");
    goto cleanup;
  }

  // KVM pruža API preko kog može da se komunicira sa njim
  // Komunikacija se vršio preko Input/Outpu sistemskih poziva
  // int ioctl(int fd, unsigned long request, ...);
  //    fd      - fajl deskriptor
  //    request - zahtev
  //    ...     - parametar koji zavisi od zahteva
  // 
  // KVM_CREATE_VM - kreira virtuelnu mašinu bez virtuelnog(ih) procesora i memorije
  vm_fd = ioctl(kvm_fd, KVM_CREATE_VM, 0);
  if (vm_fd == -1) {
    perror("Failed to create vm\n");
    goto cleanup;
  }

  mem = mmap(NULL, mem_size, PROT_READ | PROT_WRITE, MAP_SHARED | MAP_ANONYMOUS, 0, 0);
  if (mem == MAP_FAILED) {
    perror("mmap failed\n");
    goto cleanup;
  }

  // Podesavanje regiona memorije koji će se koristiti za VM
  // slot            - broj mapiranja. Definisanje novog regiona sa istim slot brojem će zameniti ovo mapiranje.
  // guest_phys_addr - Fizicka adresa kako je gost vidi.
  // memory_size     - velicina regiona.
  // userspace_addr  - početna adresa memorije.
  struct kvm_userspace_memory_region region = {
    .slot = 0,
    .guest_phys_addr = 0,
    .memory_size = mem_size,
    .userspace_addr = (uintptr_t)mem
  };
  
  // Parametar: region
  ret = ioctl(vm_fd, KVM_SET_USER_MEMORY_REGION, &region);
  if (ret < 0) {
    perror("ioctl KVM_SET_USER_MEMORY_REGION failed\n");
    goto cleanup;
  }

  // Čitanje gost programa.
  img = fopen(argv[1], "rb");
  if (img == NULL) {
    perror("Can not open binary file\n");
    goto cleanup;
  }

  // Proveravanje veličine fajla.
  fseek(img, 0, SEEK_END);
  long img_size = ftell(img);
  rewind(img);

  if (img_size < 0 || (size_t)img_size > mem_size - pc_val) {
    fprintf(stderr, "Image too large (%ld bytes)\n", img_size);
    goto cleanup;
  }

  // Popunjavanje gost memorije.
  if (fread(mem + pc_val, 1, (size_t)img_size, img) != (size_t)img_size) {
    perror("Failed to read guest image file\n");
    goto cleanup;
  }

  // Kreiranje virtuelnog CPU
  // Parametar: vCPU ID
  vcpu_fd = ioctl(vm_fd, KVM_CREATE_VCPU, 0);
  if (vcpu_fd < 0) {
    perror("Cannot create vCPU\n");
    goto cleanup;
  }

  // Dohvatanje veličine kvm_run strukture
  // Parametar: /
  int kvm_run_mmap_size = ioctl(kvm_fd, KVM_GET_VCPU_MMAP_SIZE, 0);
  if (kvm_run_mmap_size < 0) {
    perror("ioctl KVM_GET_VCPU_MMAP_SIZE failed\n");
    goto cleanup;
  }

  // Mapirati kvm_run strukturu na koju pokazuje lokalna promenljiva kvm_run
  run = (struct kvm_run *)mmap(
      NULL, kvm_run_mmap_size, PROT_READ | PROT_WRITE, MAP_SHARED, vcpu_fd, 0);
  if (run == MAP_FAILED) {
    perror("mmap kvm_run failed\n");
    goto cleanup;
  }

  struct kvm_regs regs = {0};
  struct kvm_sregs sregs;

  // Parametar: struct kvm_sregs
  ret = ioctl(vcpu_fd, KVM_GET_SREGS, &sregs);
  if (ret == -1) {
    perror("Fetching sregs failed\n");
    goto cleanup;
  }

  sregs.cs.selector = 0;
  sregs.cs.base = 0;
  sregs.ds.selector = 0;
  sregs.ds.base = 0;
  sregs.es.selector = 0;
  sregs.es.base = 0;
  sregs.fs.selector = 0;
  sregs.fs.base = 0;
  sregs.gs.selector = 0;
  sregs.gs.base = 0;
  sregs.ss.selector = 0;
  sregs.ss.base = 0;

  sregs.cr0 &= ~1u; // Clear PE bit

  ret = ioctl(vcpu_fd, KVM_SET_SREGS, &sregs);
  if (ret == -1) {
    perror("Seting sregs failed\n");
    goto cleanup;
  }

  regs.rflags = 0x2;
  regs.rip = pc_val;
  // regs.rbx = 3;

  // Parametar: struct kvm_regs
  ret = ioctl(vcpu_fd, KVM_SET_REGS, &regs); 
  if (ret == -1) {
    perror("Seting regs failed\n");
    goto cleanup;
  }

  // Pokretanje gosta i obrada izlaza 
  while(stop == 0) {
    // Parametar: /
    ret = ioctl(vcpu_fd, KVM_RUN, 0);
    if (ret == -1) {
      printf("KVM_RUN failed\n");
      goto cleanup;
    }

    switch (run->exit_reason) {
    case KVM_EXIT_IO:
      if (run->io.direction == KVM_EXIT_IO_OUT && run->io.size == 1 && run->io.port == 0x3f8 && run->io.count == 1) {
        printf("IO port: %x, data: %x\n", run->io.port, *(int *)(((char*)run)+ run->io.data_offset));
      } else {
        printf("Unhandled IO\n");
        stop = 1;
      }
      break;
    case KVM_EXIT_HLT:
      printf("KVM_EXIT_HLT\n");
      stop = 1;
      break;
    default:
      printf("Unhandled exit reason: %u\n", run->exit_reason);
      stop = 1;
      break;
    }
  }

cleanup:
  if (run != MAP_FAILED) {
    munmap(run, (size_t)kvm_run_mmap_size);
    run = MAP_FAILED;
  }

  if (vcpu_fd >= 0) {
    close(vcpu_fd);
    vcpu_fd = -1;
  }

  if (mem != MAP_FAILED) {
    munmap(mem, mem_size);
    mem = MAP_FAILED;
  }

  if (vm_fd >= 0) {
    close(vm_fd);
    vm_fd = -1;
  }

  if (kvm_fd >= 0) {
    close(kvm_fd);
    kvm_fd = -1;
  }

  if (img) {
    fclose(img);
    img = NULL;
  }

  return 0;
}
