#include <stdint.h>
#include <stdlib.h>
#include <stdio.h>
#include <time.h>

void print_i386_cpuid_caches () {
    for ( int id = 0; ; id++ ) {
        // Variables to hold the contents of the 4 registers
        uint32_t eax = 4; // get cache info
        uint32_t ebx;
        uint32_t ecx = id; // cache id
        uint32_t edx;
        
        // generates output in 4 registers eax, ebx, ecx and edx
        __asm__ (
            "cpuid":    // call i386 cpuid instruction
            // specify the output registers and variables
            // + -> register is for both reading and writing
            // = -> register is only for writing
            "+a" (eax), // contains the cpuid command code, 4 for cache query
            "=b" (ebx),
            "+c" (ecx), // contains the cache id
            "=d" (edx)
        );
        
        // See the page 3-191 of the manual.
        int cache_type = eax & 0x1F;
        
        // end of valid cache identifiers
        if ( cache_type == 0 ) {
            break;
        }
        
        char * cache_type_string;
        switch (cache_type) {
            case 1: cache_type_string  = "Data Cache"; break;
            case 2: cache_type_string  = "Instruction Cache"; break;
            case 3: cache_type_string  = "Unified Cache"; break;
            default: cache_type_string = "Unknown Type Cache"; break;
        }
        
        int cache_level                = (eax >>= 5) & 0x7;
        int cache_is_self_initializing = (eax >>= 3) & 0x1; // does not need SW initialization
        int cache_is_fully_associative = (eax >>= 1) & 0x1;
        
        // See the page 3-192 of the manual.
        // ebx contains 3 integers of 10, 10 and 12 bits respectively
        unsigned int cache_sets                     = ecx + 1;
        unsigned int cache_coherency_line_size      = (ebx & 0xFFF) + 1;
        unsigned int cache_physical_line_partitions = ((ebx >>= 12) & 0x3FF) + 1;
        unsigned int cache_ways_of_associativity    = ((ebx >>= 10) & 0x3FF) + 1;
        
        // Total cache size is the product
        size_t cache_total_size = cache_ways_of_associativity * cache_physical_line_partitions * cache_coherency_line_size * cache_sets;
        
        printf(
            "Cache ID %d:\n"
            "- Level: %d\n"
            "- Type: %s\n"
            "- Sets: %d\n"
            "- System Coherency Line Size: %d bytes\n"
            "- Physical Line partitions: %d\n"
            "- Ways of associativity: %d\n"
            "- Total Size: %zu bytes (%zu kb)\n"
            "- Is fully associative: %s\n"
            "- Is Self Initializing: %s\n"
            "\n",
            id,
            cache_level,
            cache_type_string,
            cache_sets,
            cache_coherency_line_size,
            cache_physical_line_partitions,
            cache_ways_of_associativity,
            cache_total_size, cache_total_size >> 10,
            cache_is_fully_associative ? "true" : "false",
            cache_is_self_initializing ? "true" : "false"
        );
    }
}

int get_cache_line_size ( int id ) {
    uint32_t eax = 4;
    uint32_t ebx;
    uint32_t ecx = id;
    uint32_t edx;
    
    __asm__ (
        "cpuid":
        "+a" (eax),
        "=b" (ebx),
        "+c" (ecx),
        "=d" (edx)
    );
    
    int cache_type = eax & 0x1F;
    
    if ( cache_type == 0 ) {
        return -1;
    }
    
    return (ebx & 0xFFF) + 1;
}

int main ( int argc, char **argv ) {
    print_i386_cpuid_caches ( );
    
    // int line_size = get_cache_line_size ( 0 );
    // printf ( "Cache 0 line size: %dB\n", line_size );
    
    return 0;
}