
using System.Collections.Generic;
using System.Linq;
using Microsoft.AspNetCore.Mvc;
using Microsoft.EntityFrameworkCore;
using Model;

namespace WebAPIDemo.Controllers
{

    /*
    ApiController atribut naznacava da ova klasa kao i sve njene potklase obradjuje HTTP zahteve.
    Ako je ovaj atribut prisutan neophodno je navesti putanju do resursa kome je prikacen navodjenjem
    Route atributa. Takodje, ovaj atribut dodaje neka podrazumevana pravila za povezivanje argumenata 
    metoda i vrednosti prosledjenih u zahtevu.
    */
    [ApiController] 
    /*
    Route atribut definise putanju do kontrolera. Ukoliko se kao argument navede [controller], putanje do
    kontrolera je ime klase kontrolera bez reci Controller. Medjutim, moze se navesti bilo sta. Takodje, 
    jedan kontroler se moze nalaziti na vise putanja.
    */
    [Route("[controller]")] // https://localhost:5001/User
    [Route("api/user")]     // https://localhost:5001/api/user
    /*
    Svaki kontroler koji samo obradjuje zahteve je potrebno izvesti iz klase ControllerBase. Ova klasa
    izmedju ostalog pruza usluzne metode kao sto su Ok i BadRequest kojima se moze naznaciti da je zahtev
    uspesno ili neuspesno obradjen. 
    */
    public class UserController : ControllerBase {
        private Context context;
        public UserController ( Context context ) {
            this.context = context;
        }

        /*
        HttpGet atribut naznacava da metoda obradjuje GET zahteve. 
        */
        [HttpGet] // https://localhost:5001/User ili https://localhost:5001/api/user
        public IEnumerable<User> get ( ) {
            return this.context.users.ToList ( );
        }

        /*
        HttpGet atribut prima jedan argument koji naznacava rutu do metode.
        */
        [HttpGet("getUsers")] // https://localhost:5001/User/getUsers ili https://localhost:5001/api/user/getUsers
        public IActionResult gett ( ) {
            return Ok ( this.context.users.ToList ( ) );
        }

        /*
        Jedan od nacina za prosledjivanje parametara GET zahteva je njihovo navodjenje nakon adrese. Naravno, moze se proslediti
        vise od jednog parametra. Parametri se navode nakon adrese prvo navodjenjem znaka ?, a potom i samih parametara
        po formatu ime1=vredonst1&ime2=vrednost2&....
        */
        [HttpGet("getUser")] // https://localhost:5001/User/getUser?id=5&message=Hello World!
        public IActionResult getUserGETParameters ( int? id, string message) { // int? znaci da parametar id moze da ima vrednost null, iako je primitivnog tipa.
            if ( !id.HasValue ) {
                return BadRequest ( "Invalid parameter" );
            } else {
                return Ok ( this.context.users.FirstOrDefault ( user => user.id == id ) + message );
            }
        }

        /*
        Drugi nacin prosledjivanja parametara je kroz samu putanju.
        */
        [HttpGet("getUser/{id}/{message}")] // Imena parametara u putanji mora da se poklapaju sa imenima argumenata
        public IActionResult getUserRouteParameters ( int id, string message) {
            return Ok ( message + this.context.users.FirstOrDefault ( user => user.id == id ) );
        }

        /*
        Kao i kontroler, i metoda moze imati vise putanja.
        */
        [HttpGet("trazi/{firstName}/{lastName}/{gender}")]
        [HttpGet("search/{firstName}/{lastName}/{gender}")]
        public IActionResult search ( string firstName, string lastName, string gender ) {
            IQueryable<User> query = this.context.users.Include ( user => user.actionList )
                                                        .Include ( user => user.hasRoleList )
                                                            .ThenInclude ( hasRole => hasRole.role );

            if ( firstName != null ) {
                query = query.Where ( user => user.firstName.Contains ( firstName ) );
            }
            if ( lastName != null ) {
                query = query.Where ( user => user.lastName.Contains ( lastName ) );
            }
            if ( gender != null ) {
                query = query.Where ( user => user.gender.Contains ( gender ) );
            }

            return Ok ( query.ToList ( ) );
        }

        /*
        Parametre primitivnih tipova nije moguce proslediti kroz telo POST zahteva. Stoga, prosledjuju se
        kroz URL po istom formatu kao i GET zahtevi.
        */
        [HttpPost] // POST https://localhost:5001/User?name=Jovan
        public string post ( string name ) {
            return "Hello " + name;
        }

        /*
        Promenljive kompleksih tipova se podrazumevano inicijalizuju na osnovo podataka u telu zahteva.
        Isti efekat se postize sa atributom [FromBody] ispred argumenta metode. Medjutim, potrebno je voditi 
        racuna da ne moze da postoji vise od jednog argumenta koji svoju vrednost dobija iz tela zahteva.
        */
        [HttpPost("addRole")]
        public IActionResult addRole ( Role role ) {
            // Potrebno je prvo izvuci vrednost primarnog kljuca iz baze.
            int id = this.context.roles.Max ( role => role.id );
            role.id = id + 1;
            this.context.roles.Add ( role );
            this.context.SaveChanges ( );
            return Ok ( "Role added" );
        }

        [HttpGet("getRoles")]
        public IActionResult getRoles ( ) {
            return Ok (
                this.context.roles.ToList ( )
            );
        }

        /*
        Ponekad klase modela ne zadovoljavaju uslove neophodne za obradu POST zahteva. U tom slucaju
        se prave nove klase koje ce se koristiti za skladistenje tih podataka. 
        */
        public class UserData {
            public string firstName { get; set; }
            public string lastName { get; set; }
            public string email { get; set; }
            public string gender { get; set; }
            public string username { get; set; }
            public string password { get; set; }
            public int[] roles { get; set; }
        }
        [HttpPost("addUser")]
        public IActionResult addUser ( UserData data ) {
            int id = this.context.users.Max ( user => user.id );
            IEnumerable<Role> roles = this.context.roles.Where ( role => data.roles.Contains ( role.id ) ).ToList ( );

            ICollection<HasRole> hasRoleList = new List<HasRole> ( );

            foreach ( Role role in roles ) {
                hasRoleList.Add (
                    new HasRole ( ) {
                        userId = id + 1,
                        roleId = role.id
                    }
                );
            }

            User newUser = new User ( ) {
                id        = id + 1,
                firstName = data.firstName,
                lastName  = data.lastName,
                email     = data.email,
                gender    = data.gender,
                username  = data.username,
                password  = data.password,
                hasRoleList = hasRoleList
            };

            this.context.users.Add ( newUser );
            this.context.SaveChanges ( );

            return Ok ( newUser );
        }
    }
}
