using System;
using System.Collections.Generic;
using System.ComponentModel.DataAnnotations;
using System.ComponentModel.DataAnnotations.Schema;
using Microsoft.EntityFrameworkCore;

/*
Ukoliko se podrazumevana pravila ne prate potrebno je dodatno opisati kako se domenske klase mapiraju
na tabele u bazi podataka. Jedan nacin da se to ostvari jeste koriscenjem atributa. Pravila eslikavanja SQL tipova
u C# tipove i dalje vaze.
*/
namespace UnconventionalModelWithAttributes {

    [Table("users")] // Table atribut povezuje klasu sa tabelom.
    public class Korisnik {
        [Key]           // Key atribut oznaca polje kljuc kao primarni kljuc tabele
        [Column("id")]  // Column atribut povezuje polje klase sa kolonom iz tabele
        [Required]      // Required atribut oznacava polje klase kao obavezno, odnosno da kolona u tabeli ima NOT NULL modifikator
        public int kljuc { get; set; }

        // Atribut Column pored imena kolone, dozvoljava i navodjenje tipa kolone
        [Column("firstName", TypeName = "varchar(50)")]  
        [Required]
        public string ime { get; set; }

        [Column("lastName")] // Navodjenje tipa nije neophodno ukoliko C# tip moze da se mapira na SQL tip
        [Required]
        public string prezime { get; set; }

        [Column("email", TypeName = "varchar(50)")]
        [Required]
        public string email { get; set; }

        [Column("gender", TypeName = "varchar(50)")]
        [Required]
        public string pol { get; set; }

        [Column("username", TypeName = "varchar(50)")]
        [Required]
        public string korisnickoIme { get; set; }

        [Column("password", TypeName = "varchar(50)")]
        [Required]
        public string lozinka { get; set; }

        /*
        One-to-many i Many-to-many veze se podesavaju na isti nacin. 
        */
        public ICollection<ImaUlogu> imaUloguList { get; set; }
        public ICollection<Akcija> akcijaList { get; set; }

        public override string ToString ( ) {
            return "User(" + this.kljuc + "): " + this.ime + ", " 
                                             + this.prezime + ", " 
                                             + this.email + ", " 
                                             + this.pol + ", " 
                                             + this.korisnickoIme + ", "
                                             + this.lozinka;
        }
    }

    [Table("roles")]
    public class Uloga {
        [Key]
        [Column("id")]
        [Required]
        public int kljuc { get; set; }

        [Column("name")]
        [Required]
        public string ime { get; set; }

        public ICollection<ImaUlogu> imaUloguList { get; set; }

        public override string ToString ( ) {
            return "Role(" + this.kljuc + "): " + this.ime;
        }
    }

    [Table("hasRole")]
    public class ImaUlogu {
        /*
        EF Core ne podrzava kofigurisanje kompozitnog primarnog kljuca pomocu atributa. 
        Kao u prethodnom slucaju to je neophodno odraditi u metodi OnConfiguring klase izvedene DbContext klase.
        */
        [Column("userId")]
        [Required]
        public int korisnikId { get; set; }
        public Korisnik korisnik { get; set; }
        [Column("roleId")]
        [Required]
        public int ulogaId { get; set; }
        public Uloga uloga { get; set; }

        public override string ToString ( ) {
            return "HasRole(" + this.korisnikId + ", " + this.ulogaId + ")";
        }
    }

    [Table("actions")]
    public class Akcija {
        [Key]
        [Column("id")]
        [Required]
        public int kljuc { get; set; }

        [Column("userId")]
        [Required]
        public int korisnikId  { get; set; }
        public Korisnik korisnik { get; set; }

        [Column("name")]
        [Required]
        public string ime { get; set; }

        [Column("timestamp")]
        [Required]
        public DateTime vreme { get; set; }

        public override string ToString ( ) {
            return "Action(" + this.kljuc + "): " + this.korisnikId + ", " 
                                             + this.ime + ", " 
                                             + this.vreme;
        }
    }
    public class UnconventionalContextWithAttributes : DbContext {
        // Imena tabela u ovom slucaju ne moraju da odgovaraju imenima DbSet polja.
        public DbSet<Korisnik> korisnici { get; set; }
        public DbSet<Uloga> uloge { get; set; }
        public DbSet<Akcija> akcije { get; set; }
        public DbSet<ImaUlogu> imaUlogu { get; set; }

        protected override void OnConfiguring ( DbContextOptionsBuilder builder ) {
            builder.UseMySQL ( "Server=localhost;Database=user_database;Uid=root;Pwd=root;" );
        }

        protected override void OnModelCreating ( ModelBuilder builder ) {
            /*
            Tabela HasRole ima kompozitni primarni kljuc. To je potrebno navesti u ovoj metodi.
            EF Core za razliku od EF ne podrazava pravljenje kompozitnog primarnog kljuca pomocu [Key]
            atributa.
            */
            builder.Entity<ImaUlogu> ( ).HasKey ( imaUlogu => new { imaUlogu.korisnikId, imaUlogu.ulogaId } );
        }
    }
}